/*<section title="Intro">
# NAV Quick Start Tutorial for JavaScript <br />

#### DESCRIPTION
This tutorial is a quick introduction to FairCom's JavaScript c-tree Database API (ctdb),
which is a NAV API that makes it easy to navigate your data.

It shows how to:

1. Connect to the database
2. Create a table
3. Delete records
4. Insert records
5. Look up records

It creates a console application that outputs each step to standard out.

### Why use a NAV API?
1. __NAV makes it easy to achieve unequaled performance.__

    - NAV gives you complete control over data processing.
        1. Navigate records in any order.
        2. Save record positions.
        3. Jump back to saved positions.
        4. Control how each record is locked and participates in transactions.

    - Algorithms are the secret to fast performance.
      - You are not limited to the algorithms built into SQL.
      - NAV allows you to implement any algorithm to process data,
        including algorithms for hierarchies, graphs, objects, documents, etc.

2. __NAV is the easiest way to solve complex data problems.__
    - You can walk through tough problems by navigating through the data step by step.
    - You can debug database data step by step in your favorite programming language.
    - You can leverage the power of your programming language to encapsulate complexity.

3. __NAV literally makes the database an extension of your favorite programming language.__
    - NAV makes working with data no different than working with native collections of objects.
    - NAV makes complex data processing easy to implement, debug, and maintain.


### Why use SQL?
- The easier the problem, the more ideal SQL is.
  - SQL makes it easy to do simple queries and joins.
  - SQL solves many difficult issues with prebuilt SQL syntax, such as JOIN, ORDER BY, GROUP BY, etc.
  - SQL becomes exponentially more difficult to program as data processing becomes more complicated.

## Use NAV _and_ SQL
- SQL and NAV are equal partners in the FairCom Database Engine.
- You can use both at any time (and at the same time).
- Together, they make working with data easy and fun.

<br />
<br />

Copyright (C) 2022 FairCom Corporation, 6300 West Sugar Creek Drive, Columbia, MO 65203 USA

</section> */

/* <comments for="introreferences">
## References

Change the variable values to those provided to you by FairCom. 

</comments> */

//<code title="References" name="introreferences">

let host         = 'localhost',
    sqlPort      = 6597,
    database     = 'ctreeSQL',
    userName     = 'admin',
    userPassword = 'ADMIN';

//</code>

/*<section title="Prep">
# Preparation

Create a session by sending authentication credentials 
(e.g., username and password). The server will return a 
session token that you must pass with each subsequent 
request.

</section> */

/* <comments for="connect">
## SESSION

Steps to create a session:

1) Form the JSON request body, using the credentials provided by FairCom.
2) Instantiate a new XMLHttpRequest and open the request, using the 
'POST' method. The URL must be modified with the provided hostname and port.
3) Send the request with the JSON request body from step #1.

</comments> */
//<code title="Connect" name="connect">

const me = this,
      connectRequestBody = `{
        "action": "ctreeSQL",
        "method": "connect",
        "type"  : "rpc",
        "data"  :
        [
            {
                "fcInfo":
                {
                    "nodeId":
                    {
                        "type": 0
                    },
                    "currentServer":
                    {
                        "hostOrIp"     : "${host}",
                        "sqlPortNumber": ${sqlPort},
                        "database"     : "${database}",
                        "queryTimeout" : 15
                    },
                    "userName"      : "${btoa(userName)}",
                    "userPassword"  : "${btoa(userPassword)}",
                    "autoCommit"    : true,
                    "isolationLevel": 0
                }
            }
        ]
    }`;

let xhr = me.getXhrRequest(connectRequestBody);

//</code>

/* <comments for="monitorconnection">
## Monitor Connection Events/State

You can monitor state/event changes of the XMLHttpRequest. 
In this request, we are waiting for a completion by monitoring
"onreadystatechange". When the connection is complete and we have a token,
we can then proceed to create the table. 

</comments> */
//<code title="Monitor Events" name="monitorconnection">

xhr.onreadystatechange = function({currentTarget}) {
  
  if (xhr.readyState === 4) {
     try {
        const {result} = JSON.parse(currentTarget.responseText);
        me.postMsg('Successfully connected');
        me.token = result.token;
        me.createTable();
     } catch (ignored) {
     }
  }
};

//</code>

/* <comments for="createtable">
## Create the table

1. Define request body to pass to method execScriptStatement.
2) Create the table with a defined set of fields. If the 
response is that the table already exists, you may proceed 
to the next step.

</comments> */
//<code title="Create Table" name="createtable">

function createTable() {
	const me                     = this,
		  query                  = `CREATE TABLE customer (cm_custnumb CHAR(4) PRIMARY KEY,
                                                           cm_custname VARCHAR(47),
                                                           cm_custrtng CHAR(1),
                                                           cm_custaddr VARCHAR(47),                                                           
                                                           cm_custcity VARCHAR(47),
                                                           cm_custstat CHAR(2),
                                                           cm_custzipc CHAR(9))`,
		  createTableRequestBody = me.getQueryRequestBody(query);

	let xhr = me.getXhrRequest(createTableRequestBody);

//</code>

/* <comments for="monitorcreate">
## Monitor Create Table

You can monitor state/event changes of the XMLHttpRequest. 
In this request, we are waiting for a completion by monitoring
"onreadystatechange". When the "CREATE TABLE" request is complete 
and we have a token, we can then proceed to create the table. Also, 
if the request fails because the table already exists, we can continue
to the CRUD operations. 

</comments> */
//<code title="Monitor Create Table" name="monitorcreate">

	xhr.onreadystatechange = function ({ currentTarget }) {

		if (xhr.readyState === 4) {
			try {
				const { result } = JSON.parse(currentTarget.responseText),
				      errNo      = result.error.errNo;

				// Proceed if no error or "table already exists".
				if ([ 0, -20041 ].includes(errNo)) {
				    me.postMsg(result, "CREATE TABLE response:\n");
				} else {
				    me.postMsg(result.error, "CREATE TABLE response:\n");
				}
				
				me.deleteRecords();
				
			} catch (ignored) {
			}
		}
	};
}
//</code>

/*<section title="Process">
# Processing Data
This section shows how to

1. Delete Records

2. Insert Records

3. Display Records

</section> */

/* <comments for="deleterecords">
## Delete Records

This function will truncate the table so as to not create duplicate records.

</comments> */
//<code title="Delete Records" name="deleterecords">

function deleteRecords() {
    const me = this,
          query = "TRUNCATE TABLE customer",
          truncateTableRequestBody = me.getQueryRequestBody(query); 
    
    let xhr = me.getXhrRequest(truncateTableRequestBody);

//</code>


/* <comments for="monitordelete">
## Monitor Delete Records

You can monitor state/event changes of the XMLHttpRequest. 
In this request, we are waiting for a completion by monitoring
"onreadystatechange". When the "TRUNCATE TABLE" request is complete, 
we can then proceed to create the records. If the request fails, we
will abort the process.

</comments> */
//<code title="Monitor Delete Records" name="monitordelete">

	xhr.onreadystatechange = function ({ currentTarget }) {

		if (xhr.readyState === 4) {
			try {
				const { result } = JSON.parse(currentTarget.responseText);
				const errNo      = result.error.errNo;

				// Proceed if no error (errNo 0).
				if (errNo === 0) {
			        me.postMsg(result, "TRUNCATE TABLE response:\n");
				} else {
	                me.postMsg(result.error, "TRUNCATE TABLE response:\n");
				}
				
			    me.insertRecords();
			    
			} catch (ignored) {
			}
		}
	};
}
//</code>

/* <comments for="insertrecords">
## Insert Records

Insert records. Records can either be inserted 
one at a time or in groups, as shown in this example.

</comments> */

//<code title="Insert Records" name="insertrecords">

function insertRecords() {
    const me = this,
          query = `INSERT INTO customer (cm_custnumb, cm_custname, cm_custrtng, 
                                         cm_custaddr, cm_custcity, cm_custstat, cm_custzipc)
                   VALUES ( '1000', 'Bryan Williams', '1', '2999 Regency',      'Orange',   'CA', '92867' ),
                          ( '1001', 'Michael Jordan', '1', '13 Main',           'Hartford', 'CT', '61434' ),
                          ( '1002', 'Joshua Brown',   '1', '4356 Cambridge',    'Atlanta',  'GA', '73677' ),
                          ( '1003', 'Keyon Dooling',  '1', '19771 Park Avenue', 'Columbia', 'MO', '10034' )`,
          insertRecordsRequestBody = me.getQueryRequestBody(query); 
    
    let xhr = me.getXhrRequest(insertRecordsRequestBody);

//</code>

/* <comments for="monitorinsert">
## Monitor Insert Records

You can monitor state/event changes of the XMLHttpRequest. 
In this request, we are waiting for a completion by monitoring
"onreadystatechange". 

When the "INSERT INTO" request is complete, 
we can then proceed to display the records. If the request fails, we
will abort the process.

</comments> */

//<code title="Monitor Insert Records" name="monitorinsert">

    xhr.onreadystatechange = function ({ currentTarget }) {

        if (xhr.readyState === 4) {
            try {
                const { result } = JSON.parse(currentTarget.responseText);
                const errNo      = result.error.errNo;

                // Proceed if no error (errNo 0).
                if (errNo === 0) {
                    me.postMsg(result, "INSERT INTO response:\n");
                } else {
                    me.postMsg(result.error, "INSERT INTO response:\n");
                }
                
                me.displayRecords();
                
            } catch (ignored) {               
            }
        }
    };
}

//</code>

/* <comments for="displayrecords">
## Display Records

Display Records that were just inserted. 

</comments> */

//<code title="Display Records" name="displayrecords">

function displayRecords() {
    const me = this,
          query = `SELECT * FROM customer`,
          displayRecordsRequestBody = me.getQueryRequestBody(query); 
    
    let xhr = me.getXhrRequest(displayRecordsRequestBody);

//</code>

/* <comments for="monitordisplay">
## Monitor Insert Records

You can monitor state/event changes of the XMLHttpRequest. 
In this request, we are waiting for a completion by monitoring
"onreadystatechange". 

When the "SELECT * FROM" request is complete, 
we can then proceed to disconnect the session.

</comments> */

//<code title="Monitor Display Records" name="monitordisplay">

    xhr.onreadystatechange = function ({ currentTarget }) {

        if (xhr.readyState === 4) {
            try {
                const { result } = JSON.parse(currentTarget.responseText);
                const errNo      = result.error.errNo;

                // Proceed if no error (errNo 0).
                if (errNo === 0) {
                    me.postMsg(result, "SELECT * FROM response:\n");
                } else {
                    me.postMsg(result.error, "SELECT * FROM response:\n");
                }
                
                me.disconnect();
                
            } catch (ignored) {               
            }
        }
    };
}

//</code>

/* <comments for="disconnect">
## Close Session

A disconnect request is sent to the server so that it knows 
to not expect any further requests from this client. 

</comments> */
//<code title="Disconnect" name="disconnect">

function disconnect() {
    const me = this,
          disconnectRequestBody = `{
            "action": "ctreeSQL",
            "method": "disconnect",
            "type"  : "rpc",
            "data"  :
            [
                {
                    "token" : "${me.token}",
                    "fcInfo":
                    {
                        "nodeId":
                        {
                            "type": 0
                        },
                        "currentServer":
                        {
                            "hostOrIp"     : "${host}",
                            "sqlPortNumber": ${sqlPort},
                            "database"     : "${database}",
                            "queryTimeout" : 15
                        },
                        "autoCommit"    : true,
                        "isolationLevel": 0
                    }
                }
            ]
        }`;
    
    let xhr = me.getXhrRequest(disconnectRequestBody);

//</code>

/* <comments for="monitordisconnection">

## Monitor Disconnection Events/State

You can monitor state/event changes of the XMLHttpRequest. 
In this request, we are waiting for a completion by monitoring
"onreadystatechange". When the disconnection is complete,
the session is cleaned up.

</comments> */
//<code title="Monitor Disconnect" name="monitordisconnection">

    xhr.onreadystatechange = function({currentTarget}) {
      
      if (xhr.readyState === 4) {
         try {           
            if (currentTarget.status === 200) {
                me.postMsg("Successfully disconnected.", "DISCONNECTION response:\n");
            } else {
                let responseObj = JSON.parse(currentTarget.responseText);
                me.postMsg(responseObj.message, "DISCONNECTION response:\n");
            }
         } catch (ignored) {
         }
      }
    };
}

//</code>

/* <comments for="utilfunctions">

## Utility Functions

Reusable functions leveraged by main code flow.

</comments> */

//<code title="Utility Functions" name="utilfunctions">

function getQueryRequestBody(query) {
   const me  = this;
   
   return `{
              "action": "ctreeSQL",
              "method": "execScriptStatement",
              "type"  : "rpc",
              "data"  :
              [
                  {
                      "token" : "${me.token}",
                      "fcInfo":
                      {
                          "query"        : "${query}",
                          "maxRows"      : "1000",
                          "noTruncate"   : false,
                          "currentServer":
                          {
                              "hostOrIp"     : "${host}",
                              "sqlPortNumber": ${sqlPort},
                              "database"     : "${database}",
                              "queryTimeout" : 15
                          },
                          "time": false
                      }
                  }
              ]
          }`;
}

function getXhrRequest(requestBody) {
   const me = this;
   
   let xhr = new XMLHttpRequest();
   xhr.open('POST', `${location.origin}/SqlExplorer/ctSQLExplorerRouter`);
   xhr.send(requestBody);
   
   return xhr;
}

function hasJsonStructure(msg) {
    if (typeof msg !== "string") return false;
    
    try {
        const result = JSON.parse(msg),
        type = Object.prototype.toString.call(result);
        
        return type === "[object Object]" || type === "[object Array]";
    } catch (err) {
        return false;
    }
}

/**
 *  This function is only needed in this Tutorials context where the user
 *  code is being executed in a web worker and the results need to be transmitted
 *  back to the Tutorials app so it can log to the results panel.
 */
function postMsg(msg, title = "") {
    const msgType = Object.prototype.toString.call(msg);

    switch (msgType) {
        case "[object Object]" :
            postMessage(title + JSON.stringify(msg, null, 4) + '\n');
            break;
        case "[object String]" :
            if (hasJsonStructure(msg)) {
                try {
                    postMessage(title + JSON.stringify(JSON.parse(msg), null, 4));
                } catch (err) {
                    postMessage(title + msg + '\n');
                }
            } else {
                postMessage(title + msg + '\n');
            }
            break;
        default : 
            postMessage(title + msg + '\n');
    }
}
//</code>
