/*<section title="Intro">
# NAV Quick Start Tutorial for C

#### DESCRIPTION
This tutorial is a quick introduction to FairCom's c-tree Database API (ctdb), 
which is a NAV API that makes it easy to navigate your data.

It shows how to:
1. Connect to the database
2. Create a table
3. Insert records
4. Look up records

It creates a console application that outputs each step to standard out.

### Why use a NAV API?
1. __NAV makes it easy to achieve unequaled performance.__
    - NAV gives you complete control over data processing.
        1. Navigate records in any order.
        2. Save record positions.
        3. Jump back to saved positions.
        4. Control how each record is locked and participates in transactions.
    - Algorithms are the secret to fast performance.
      - You are not limited to the algorithms built into SQL. 
      - NAV allows you to implement any algorithm to process data, 
        including algorithms for hierarchies, graphs, objects, documents, etc.
        
2. __NAV is the easiest way to solve complex data problems.__
    - You can walk through tough problems by navigating through the data step by step.
    - You can debug database data step by step in your favorite programming language.
    - You can leverage the power of your programming language to encapsulate complexity.  
    
3. __NAV literally makes the database an extension of your favorite programming language.__
    - NAV makes working with data no different than working with arrays of structures.
    - NAV makes complex data processing easy to implement, debug, and maintain.
  

### Why use SQL?
- The easier the problem, the more ideal SQL is.
  - SQL makes it easy to do simple queries and joins.
  - SQL solves many difficult issues with prebuilt SQL syntax, such as JOIN, ORDER BY, GROUP BY, etc.
  - SQL becomes exponentially more difficult to program as data processing becomes more complicated.
  
## Use NAV _and_ SQL
- SQL and NAV are equal partners in the FairCom Database Engine.
- You can use both at any time (and at the same time).
- Together, they make working with data easy and fun.

<br />
<br />
<br />

Copyright (C) 2020 by FairCom Corporation, <br />
6300 West Sugar Creek Drive, Columbia, MO 65203 USA

</section> */



/* <comments for="references">

## Adding NAV to your project
1. Include ctdbsdk.h to access FairCom's driver functions.
2. For convenience, this tutorial creates four global handle variables to manage a database session, database, table, and records.
   - __``hSession``__
   - __``hDatabase``__
   - __``hTable``__
   - __``hRecord``__

<br />

<div style="text-align: center; line-height: normal;">

## You can now leverage the full power of the world's leading _binary structure database_

</div>

#### NOTES
1. Include the __``mtclient.dll``__ in your project and make sure it is distributed with your application.
   It is FairCom's client driver between your application and the FairCom Database engine. <br /><br />
2. Reference __``libeay32.lib``__ and __``ssleay32.lib``__ in your project. These provide encrypted communications with FairCom's client driver.
   - The cryptography files are located in the folder: <br /> __``<...database...>\drivers\ctree.drivers\lib\License.Lib\openssl\winX64\Microsoft Visual Studio ____``__
   - The DLL file is located in the folder: <br /> __``<...database...>\drivers\ctree.drivers\bin``__



</comments> */
//<code title="References" name="references">

#ifdef _WIN32_WCE
#undef UNICODE
#undef _UNICODE
#define main my_main
#endif

#include "ctdbsdk.h"

#define END_OF_FILE INOT_ERR

CTHANDLE hSession;
CTHANDLE hDatabase;
CTHANDLE hTable;
CTHANDLE hRecord;

#ifdef PROTOTYPE
VOID Initialize(VOID), Define(VOID), Manage(VOID), Done(VOID);
VOID Add_Records(VOID), Display_Records(VOID);
VOID Delete_Records(CTHANDLE), Check_Table_Mode(CTHANDLE);
VOID Handle_Error(CTSTRING);
#else
VOID Initialize(), Define(), Manage(), Done();
VOID Add_Records(), Display_Records();
VOID Delete_Records(), Check_Table_Mode();
VOID Handle_Error();
#endif

#ifdef ctPortAppleiOS
#define main my_main
#endif


//</code>

/* <comments for="main">
## MAIN

The tutorial is organized into four functions:

1. __``Initialize()``__
   - Connects to the database.

2. __``Define()``__
   - Opens an existing table or creates a new table.

3. __``Manage()``__
   - Deletes existing records (if any).
   - Adds records to the table.
   - Queries the table and outputs the results to stdout.

4. __``Done()``__
   - Disconnects from the database.

When the tutorial has finished running, 
you can press the ENTER key to close the console.


</comments> */
//<code title="Main" name="main">

#ifdef PROTOTYPE
NINT main (NINT argc, pTEXT argv[])
#else
NINT main (argc, argv)
NINT argc;
TEXT argv[];
#endif
{
   Initialize();

   Define();

   Manage();

   Done();

#ifndef ctPortAppleiOS
   printf("\nPress <ENTER> key to exit . . .\n");
#ifndef ctPortWINCE
   getchar();
#endif
#endif

   return(0);
}


//</code>

/*<section title="Prep">
# Preparation
This section shows how to connect to the database and create tables.

<br />

<div style="line-height: normal;">

## _Use your favorite programming language to completely control all database functions_

</div>

</section> */

/* <comments for="initialize">
## Initialize

__``Initialize()``__ does the following:
1. Starts an embedded FairCom Database Engine.
2. Initializes global session object that stores session state.
3. Logs into the session.

#### NOTES
1. __Logging in__
   - The default user name and password are both __``ADMIN``__
   - The default name for the database server is __``FAIRCOMS``__

2. __Starting the database engine__
   - Starting the database engine from within your application is only required 
     when you embed the database into your application
     by compiling it in or by linking to it statically or dynamically.


3. __Embedding the database in your application__
   - The FairCom Database Engine has the unique ability to be embedded in an application.
   - The entire database engine can run as a DLL in the application.
   - This is useful for creating self-contained applications for:
      - Devices
      - Desktop Applications
      - Mobile apps
   - When the application connects to an external FairCom Database Engine (as in an N-tier Application Model), 
     comment out the following line of code:
      - __``ctdbStartDatabaseEngine()``__
  

<br />

<div style="text-align: center; line-height: normal;">
  
## _You can connect to many FairCom databases and simultaneously "remote control" all of them_

</div>

</comments> */
//<code title="Connect to Database" name="initialize">

#ifdef PROTOTYPE
VOID Initialize(VOID)
#else
VOID Initialize()
#endif
{
   CTDBRET  retval;
   
   printf("INIT\n");

   if ((retval = ctdbStartDatabaseEngine()))
      Handle_Error("Initialize(): ctdbStartDatabaseEngine()");

   if ((hSession = ctdbAllocSession(CTSESSION_CTREE)) == NULL)
      Handle_Error("Initialize(): ctdbAllocSession()");

   hDatabase = hSession;

   printf("\tLogon to server...\n");
   if (ctdbLogon(hSession, "FAIRCOMS", "ADMIN", "ADMIN"))
      Handle_Error("Initialize(): ctdbLogon()");
}
//</code>

/* <comments for="define">
## Define

__``Define()``__ shows how to:
1. Initialize empty global table handle for later use.
2. Open the table, if it exists. 
3. Otherwise create the table and then open it.

#### NOTES

1. A typical pattern in NAV is to first open a FairCom Engine Object, such as a database or table,
and if it doesn't exist, create it.
   
2. Another common pattern in NAV is to configure a FairCom Engine Object and then apply it.
   - For example, the __``ctdbAddField()``__ method populates the __``hTable handle``__ with field definitions.
     and then the __``ctdbCreateTable()``__ method creates the table containing those fields.

3. Another pattern is for the NAV API to return an error code when something does not work.
   - Everything in a database is stateful and all state changes follow rules. 
   - Many database operations change state and can potentially break rules. 
   - When an operation breaks a rule, such as trying to add a table or a record that already exists,
     the operation returns an error code to indicate that it did not change state.
   - Thus, NAV code wraps each major operation in code that checks return codes.

<br />

<div style="text-align: center; line-height: normal;">

## _NAV is a set of simple and intuitive functions_

</div>


</comments> */
//<code title="Create a Table" name="define">

#ifdef PROTOTYPE
VOID Define(VOID)
#else
VOID Define()
#endif
{
   CTHANDLE hField1, hField2, hField3, hField4;
   CTHANDLE hField5, hField6, hField7;

   printf("DEFINE\n");

   if ((hTable = ctdbAllocTable(hDatabase)) == NULL)
      Handle_Error("Define(); ctdbAllocTable()");

   printf("\tOpen table...\n");
   if (ctdbOpenTable(hTable, "custmast", CTOPEN_NORMAL))
   {
      printf("\tAdd fields...\n");
      hField1 = ctdbAddField(hTable, "cm_custnumb", CT_FSTRING, 4);
      hField2 = ctdbAddField(hTable, "cm_custzipc", CT_FSTRING, 9);
      hField3 = ctdbAddField(hTable, "cm_custstat", CT_FSTRING, 2);
      hField4 = ctdbAddField(hTable, "cm_custratg", CT_FSTRING, 1);
      hField5 = ctdbAddField(hTable, "cm_custname", CT_STRING, 47);
      hField6 = ctdbAddField(hTable, "cm_custaddr", CT_STRING, 47);
      hField7 = ctdbAddField(hTable, "cm_custcity", CT_STRING, 47);

      if (!hField1 || !hField2 || !hField3 || !hField4 ||
          !hField5 || !hField6|| !hField7)
         Handle_Error("Define(); ctdbAddField()");

      printf("\tCreate table...\n");
      if (ctdbCreateTable(hTable, "custmast", CTCREATE_NORMAL))
         Handle_Error("Define(); ctdbCreateTable()");

      if (ctdbOpenTable(hTable, "custmast", CTOPEN_NORMAL))
         Handle_Error("Define(); ctdbOpenTable()");
   }
   else
      Check_Table_Mode(hTable);
}
//</code>


/*<section title="Process">
# Processing Data
This section shows how to 

1. Delete Records
2. Insert Records
3. Display Records

<br />

## _NAV eliminates double processing of your data_
  
  - A SQL database loops through your data and then returns the data so your application can loop through it again.
  - In NAV, your application loops through the data once by "remote controlling" the database.




</section> */

/* <comments for="manage">
## Process Data

Manage() calls a separate function to do each of the following:
1. Delete records
2. Insert records
3. Display records

<br />
  
## _Literally extend your app directly into the database_

</comments> */
//<code title="Process Data" name="manage">

#ifdef PROTOTYPE
VOID Manage(VOID)
#else
VOID Manage()
#endif
{
   printf("MANAGE\n");

   if ((hRecord = ctdbAllocRecord(hTable)) == NULL)
      Handle_Error("Manage(): ctdbAllocRecord()");

   Delete_Records(hRecord);

   Add_Records();

   Display_Records();
}
//</code>


/* <comments for="DeleteRecords">
## Delete Records

__``Delete_Records()``__ shows how to 
1. Find the first record in a table.
2. Delete all records in a table one by one.

<br />

#### Algorithm
  - Find the first record
  - When a record is found: 
    - Delete it
    - Move to next record
  - Repeat

<br />

#### NOTES
 - Working with data is the same as working with arrays of structures. 
   Your code controls every step.
 - The FairCom Database Engine is designed specifically to give your application complete control,
   and do it with unrivaled speed.
 - Everything about the database engine is designed to navigate data:
   - The indexes have a special structure to make  
     __``ctdbFirstRecord()``__, __``ctdbNextRecord()``__, __``ctdbPrevRecord()``__, and  __``ctdbLastRecord()``__ very fast.
   - The communication protocols are highly optimized to minimize latency and marshalling. 

<br />

<div style="text-align: center; line-height: normal;">

## _The database becomes an extension of the application &mdash; doing exactly what the application tells it to do_

</div>

</comments> */
//<code title="Delete Records" name="DeleteRecords">

#ifdef PROTOTYPE
VOID Delete_Records(CTHANDLE hRecord)
#else
VOID Delete_Records(hRecord)
CTHANDLE hRecord;
#endif
{
   CTDBRET  retval;
   CTBOOL   empty;

   printf("\tDelete records...\n");

   empty = NO;
   retval = ctdbFirstRecord(hRecord);
   if (retval != CTDBRET_OK)
   {
      if (retval == END_OF_FILE)
         empty = YES;
      else
         Handle_Error("Delete_Records(): ctdbFirstRecord()");
   }

   while (empty == NO)
   {
      if (ctdbDeleteRecord(hRecord))
         Handle_Error("Delete_Records(): ctdbDeleteRecord()");

      retval = ctdbNextRecord(hRecord);
      if (retval != CTDBRET_OK)
      {
         if (retval == END_OF_FILE)
            empty = YES;
         else
            Handle_Error("Delete_Records(): ctdbNextRecord()");
      }
   }
}
//</code>

/* <comments for="AddRecords">
## Add Records

__``Add_Records()``__ shows how to insert records into a table.
1. It uses the __``CUSTOMER_DATA``__ structure to hold customer data.
2. It creates an array of __``CUSTOMER_DATA``__ structures and populates them with data.
3. It uses that array of structures to insert records into the "custmast" table.
<br />

#### NOTES

This example illustrates how NAV makes the FairCom Database Engine an extension of your code.
In fact, in C you can literally write the binary data in C Structs directly into records 
and read binary data out of records directly into C Structs. This completely eliminates data transformation.

In all programming languages, the c-tree Database API makes it easy and fast 
to put a buffer directly into a record and vice-versa.
It is fast and efficient because under the hood, its driver writes data into a binary buffer, and 
that buffer goes directly into and out of the database without translation.
In other words, the application does all the work of mapping and formatting data,
and the database simply processes all data as simple binary data stored in records.

This approach distributes the processing load between the client and the database. 
This is much more effective than SQL databases because a SQL database must do all the work 
of translating client data into proprietary internal database structures and vice-versa.

<br />

<div style="text-align: center; line-height: normal;">

## _Achieve extreme speed by minimizing and distributing data&nbsp;transformation across app servers_

</div>

</comments> */
//<code title="Add Records" name="AddRecords">

typedef struct {
   CTSTRING number, zipcode, state, rating, name, address, city;
} CUSTOMER_DATA;

CUSTOMER_DATA data[] = {
   "1000", "92867", "CA", "1", "Bryan Williams", "2999 Regency",      "Orange",
   "1001", "61434", "CT", "1", "Michael Jordan", "13 Main",           "Harford",
   "1002", "73677", "GA", "1", "Joshua Brown",   "4356 Cambridge",    "Atlanta",
   "1003", "10034", "MO", "1", "Keyon Dooling",  "19771 Park Avenue", "Columbia"
};

#ifdef PROTOTYPE
VOID Add_Records(VOID)
#else
VOID Add_Records()
#endif
{
   CTDBRET  retval;
   CTSIGNED i;
   CTSIGNED nRecords = sizeof(data) / sizeof(CUSTOMER_DATA);

   printf("\tAdd records...\n");

   for (i = 0; i < nRecords; i++)
   {
      ctdbClearRecord(hRecord);

      retval = 0;
      retval |= ctdbSetFieldAsString(hRecord, 0, data[i].number);
      retval |= ctdbSetFieldAsString(hRecord, 1, data[i].zipcode);
      retval |= ctdbSetFieldAsString(hRecord, 2, data[i].state);
      retval |= ctdbSetFieldAsString(hRecord, 3, data[i].rating);
      retval |= ctdbSetFieldAsString(hRecord, 4, data[i].name);
      retval |= ctdbSetFieldAsString(hRecord, 5, data[i].address);
      retval |= ctdbSetFieldAsString(hRecord, 6, data[i].city);

      if (retval)
         Handle_Error("Add_Records(): ctdbSetFieldAsString()");

      if (ctdbWriteRecord(hRecord))
         Handle_Error("Add_Records(): ctdbWriteRecord()");
   }
}
//</code>

/* <comments for="DisplayRecords">
## Display Records

__``Display_Records()``__ shows how to retrieve records from a table,
process them, and output results.
1. __``ctdbFirstRecord()``__ retrieves the first record.
2. __``ctdbNextRecord()``__ retrieves the next record.
3. Two fields are retrieved as a string and displayed.

#### NOTES

__``ctdbFirstRecord()``__ and __``ctdbNextRecord()``__ process data in table or index order:
 
- Table order returns records very quickly in non-deterministic order.
  - This is the order used in __``Display_Records()``__.
  - This is the fastest way to traverse records.
  
- Index order returns records in the order the index stores them.
  - Index order always retrieves records in sorted order.
  - Indexes are ordered by one or more fields in the record.
    Each field can be independently sorted in ascending or descending order.
  - The FairCom Database Engine has a unique index structure so that it can walk records quickly.
  - As long as your table has an index, your application does not need to do the expensive operation of sorting records.
  - Create only the indexes you need because each index requires the database to do extra work. 
  
- The FairCom Database Engine creates two indexes by default: __``ROWID``__ and __``RECBYT``__
  1. __``ROWID``__ orders records by an automatically generated ascending integer
     - Thus, __``ROWID``__ order is always in original insert order
     - Because an index can be navigated forward or backward,
       you can use the ROWID index to walk data in oldest to newest order or vice-versa.
  2. __``RECBYT``__ orders records by their offset in the file.
      - Thus, __``RECBYT``__ order is always the order records are stored in a file.
      - This index is used to track the physical byte offset
        of records in files that contain variable-length records.
      - It is not needed when all table fields have a fixed length.
<br />
<br />

<div style="text-align: center; line-height: normal;">

## _Leverage FairCom's unique ability to process fixed-length records extremely quickly_ 

</div>

</comments> */
//<code title="Display Records" name="DisplayRecords">

#ifdef PROTOTYPE
VOID Display_Records(VOID)
#else
VOID Display_Records()
#endif
{
   CTDBRET  retval;
   TEXT     custnumb[4+1];
   TEXT     custname[47+1];

   printf("\tDisplay records...");

   retval = ctdbFirstRecord(hRecord);
   if (retval != CTDBRET_OK)
      Handle_Error("Display_Records(): ctdbFirstRecord()");

   while (retval != END_OF_FILE)
   {
      retval = 0;
      retval |= ctdbGetFieldAsString(hRecord, 0, custnumb, sizeof(custnumb));
      retval |= ctdbGetFieldAsString(hRecord, 4, custname, sizeof(custname));
      if (retval)
         Handle_Error("Display_Records(): ctdbGetFieldAsString()");

      printf("\n\t\t%-8s%10s\n",custnumb, custname);

      retval = ctdbNextRecord(hRecord);
      if (retval == END_OF_FILE)
         break;

      if (retval != CTDBRET_OK)
         Handle_Error("Display_Records(): ctdbNextRecord()");
   }
}
//</code>


/*<section title="Cleanup">
# Cleanup
This section shows how to close tables and disconnect from the database.

<br />
<div style="text-align: center; line-height: normal;">

## Release database objects <br /> when your code is done with them <br /> &mdash; just like any other resource in your code &mdash; 

</div>

</section> */


/* <comments for="done">
## Done

__``Done()``__ shows how to 

1. Close an open table.
2. Log out from the FairCom Database Engine.
3. Stop an embedded FairCom Database Engine.

<br />

#### NOTES
- You only need to stop the database engine when the database is embedded in the application.

</comments> */
//<code title="Disconnect from Database" name="done">

#ifdef PROTOTYPE
VOID Done(VOID)
#else
VOID Done()
#endif
{
   printf("DONE\n");

   printf("\tClose table...\n");
   if (ctdbCloseTable(hTable))
      Handle_Error("Done(): ctdbCloseTable()");

   printf("\tLogout...\n");
   if (ctdbLogout(hSession))
      Handle_Error("Done(): ctdbLogout()");

   ctdbFreeRecord(hRecord);
   ctdbFreeTable(hTable);
   ctdbFreeSession(hSession);

   ctdbStopDatabaseEngine();
}
//</code>

/*<section title="General">
# General
This section contains helper functions:
1. to handle exceptions.
2. to change the table transaction processing mode.

<br />
<br />

<div style="line-height: normal;">

## _Encapsulate and extend the NAV API using your favorite programming language features_

</div>

</section> */


/* <comments for="handleError">
## Handling General Exceptions

__``Handle_Error()``__ is invoked by all of the error catching statements in this tutorial, and does the following:
1. Writes an unexpected error to standard out.
2. Prompts the user to press ENTER.
3. Waits for the user to press ENTER.
4. Exits the application with an error code of 1.

</comments> */
//<code title="Handling General Exceptions" name="handleError">


#ifdef PROTOTYPE
VOID Handle_Error(CTSTRING errmsg)
#else
VOID Handle_Error(errmsg)
CTSTRING errmsg;
#endif
{
   printf("\nERROR: [%d] - %s \n", ctdbGetError(hSession), errmsg);
   printf("*** Execution aborted *** \n");
#ifndef ctPortAppleiOS
   printf("Press <ENTER> key to exit...");
#endif

   ctdbLogout(hSession);

   ctdbFreeRecord(hRecord);
   ctdbFreeTable(hTable);
   ctdbFreeSession(hSession);

#ifndef ctPortAppleiOS
   getchar();
#endif

   exit(1);
}
//</code>

/* <comments for="CheckTableMode">
## Changing Table Transaction Mode

__``Check_Table_Mode()``__ is a helper function 
used during table creation to disable transaction processing.
It illustrates how you can control the transaction processing characteristics of each individual table.

Transaction control ensures full ACID durability and, like all other ACID-compliant SQL databases, 
it comes with the performance cost of writing transactions into transaction logs on disk.

Turning off transaction control increases the speed of all database inserts, updates, and deletes, 
but it also increases the risk of data loss if the hardware or software fails unexpectedly.

### Function Process
- It gets the create mode of a table.
- If the table is under transaction control, 
  it updates the table's create mode to not be under transaction control.
- A table's creation modes are a set of bit flags. This code turns off the "transaction&nbsp;control" bit flag.

<br />

### NOTES
- Transaction control is turned off in this tutorial to illustrate one of many low-level table features 
  that you can individually control in each table. 
- The tutorial works equally well transaction control turned on or off.

<div style="text-align: center; line-height: normal;">

## _NAV makes it easy <br /> to take complete control of your data_

</div>
  
</comments> */
//<code title="Changing Table Transaction Mode" name="CheckTableMode">

#ifdef PROTOTYPE
VOID Check_Table_Mode(CTHANDLE hTable)
#else
VOID Check_Table_Mode(hTable)
CTHANDLE hTable;
#endif
{
   CTCREATE_MODE mode;

   mode = ctdbGetTableCreateMode(hTable);

   if ((mode & CTCREATE_TRNLOG))
   {
      mode ^= CTCREATE_TRNLOG;
      if (ctdbUpdateCreateMode(hTable, mode) != CTDBRET_OK)
         Handle_Error("Check_Table_Mode(); ctdbUpdateCreateMode");
   }
}
//</code>
