 /*
 * SPDX-License-Identifier: AGPL-3.0-or-later
 */
package it.finanze.sanita.gva.validator;

 import static it.finanze.sanita.gva.dto.XPathAndValuesDTO.buildPathValue;

import java.util.ArrayList;
import java.util.List;

import it.finanze.sanita.gva.dto.ErrorUCTDTO;
import it.finanze.sanita.gva.enums.CVACErrorEnum;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;


 @NoArgsConstructor(access = AccessLevel.PRIVATE)
 public final class CVACValidator extends DocumentTypeValidator {


 	public static List<ErrorUCTDTO> validateUC18(String xml) {
 		List<ErrorUCTDTO> errors = new ArrayList<>();

 		ValidationXML.equalIC(errors, CVACErrorEnum.T000, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale]", "classCode", "DOCSET");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T001, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale]", "moodCode", "EVN");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T002, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.3.2");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T003, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > code", "code", "11369-6");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T004, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > code", "codeSystem", "2.16.840.1.113883.6.1");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T005, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > code", "codeSystemName", "LOINC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T006, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > code", "displayName", "Storia delle immunizzazioni");
 		ValidationXML.equalTAG(errors, CVACErrorEnum.T007, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > title", "Certificato Vaccinale");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T008, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "classCode", "SBADM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T009, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "moodCode", "EVN");
 		ValidationXML.checkSectionByTemplateId(errors, CVACErrorEnum.T010, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "", "2.16.840.1.113883.2.9.10.1.11.4.1", false, buildPathValue("statusCode", "code", "completed"));
 		ValidationXML.checkSectionByTemplateId(errors, CVACErrorEnum.T011, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "", "2.16.840.1.113883.2.9.10.1.11.4.2", false, buildPathValue("statusCode", "code", "cancelled"));
 		ValidationXML.equalIC(errors, CVACErrorEnum.T012, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > routeCode", "codeSystem", "2.16.840.1.113883.5.112");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T013, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > routeCode", "codeSystemName", "RouteOfAdministration");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T014, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > approachSiteCode", "codeSystem", "2.16.840.1.113883.5.1052");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T015, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > approachSiteCode", "codeSystemName", "ActSite");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T016, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > consumable", "typeCode", "CSM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T017, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystem", "2.16.840.1.113883.2.9.6.1.5");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T018, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystemName", "AIC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T019, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code > translation", "codeSystem", "2.16.840.1.113883.6.73");
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T023, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.1", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.3", buildPathValue("code", "code","30973-2"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.6.1"), buildPathValue(null, "typeCode","SUBJ"), buildPathValue(null, "inversionInd","true"), buildPathValue("observation", "moodCode", "EVN"), buildPathValue("statusCode", "code", "completed"));
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T024, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.1", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.4", buildPathValue("code", "code", "59778-1", "30981-5", "30980-7", "59777-3", "59778-1"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.4.642.3.308"), buildPathValue(null, "typeCode", "REFR"), buildPathValue(null, "inversionInd","false"), buildPathValue("observation", "moodCode", "EVN"), buildPathValue("statusCode", "code", "completed"));
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T025, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.1", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.5", buildPathValue("code", "code","95715-9"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.6.1"), buildPathValue(null, "typeCode","RSON"), buildPathValue("code", "codeSystemName", "LOINC"), buildPathValue("observation", "moodCode", "EVN"), buildPathValue("statusCode", "code", "completed"));
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T026, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.1", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.6", buildPathValue("code", "code","59785-6"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.6.1"), buildPathValue(null, "typeCode","RSON"), buildPathValue("code", "codeSystemName", "LOINC"), buildPathValue("observation", "moodCode", "EVN"), buildPathValue("statusCode", "code", "completed"));
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T027, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.1", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.8", buildPathValue("code", "code","31044-1"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.6.1"), buildPathValue(null, "typeCode","CAUS"), buildPathValue("code", "codeSystemName", "LOINC"), buildPathValue("observation", "moodCode", "EVN"), buildPathValue("statusCode", "code", "completed"));
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T028, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.1", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.9", buildPathValue("code", "code","75323-6"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.6.1"), buildPathValue(null, "typeCode","RSON"), buildPathValue("code", "codeSystemName", "LOINC"), buildPathValue("observation", "moodCode", "EVN"), buildPathValue("statusCode", "code", "completed"));
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T029, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.2", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.10", buildPathValue("code", "code","85714-4"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.6.1"), buildPathValue(null, "typeCode","RSON"), buildPathValue("code", "displayName", "Ragione esonero/omissione o differimento"), buildPathValue("code", "codeSystemName", "LOINC"), buildPathValue("observation", "moodCode", "EVN"), buildPathValue("statusCode", "code", "completed"));
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T030, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.2", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.7", buildPathValue("code", "code","59784-9"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.6.1"), buildPathValue(null, "typeCode","RSON"), buildPathValue("code", "displayName", "Malattia con immunità presunta"), buildPathValue("code", "codeSystemName", "LOINC"), buildPathValue("observation", "moodCode", "EVN"), buildPathValue("statusCode", "code", "completed"));
 		ValidationXML.equalIC(errors, CVACErrorEnum.T031, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > entryRelationship > observation > statusCode", "code", "completed");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T032, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "classCode", "SBADM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T033, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "moodCode", "EVN");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T034, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > consumable", "typeCode", "CSM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T035, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystem", "2.16.840.1.113883.2.9.6.1.5");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T036, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystemName", "AIC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T037, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code > translation", "codeSystem", "2.16.840.1.113883.6.73");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T038, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > participant", "typeCode", "LOC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T039, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > entryRelationship", "typeCode", "SUBJ", "RSON");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T040, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > entryRelationship > observation", "moodCode", "EVN");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T041, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > entryRelationship > observation > code", "codeSystemName", "LOINC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T042, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > entryRelationship > observation > statusCode", "code", "completed");

 		System.out.println(getErrors(errors));

 		return errors;
 	}


 	public static List<ErrorUCTDTO> validateUC19(String xml) {
 		List<ErrorUCTDTO> errors = new ArrayList<>();

 		ValidationXML.equalIC(errors, CVACErrorEnum.T043, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale]", "classCode", "DOCSET");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T044, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale]", "moodCode", "EVN");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T045, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.3.2");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T046, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > code", "code", "11369-6");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T047, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > code", "codeSystem", "2.16.840.1.113883.6.1");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T048, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > code", "codeSystemName", "LOINC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T049, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > code", "displayName", "Storia delle immunizzazioni");
 		ValidationXML.equalTAG(errors, CVACErrorEnum.T050, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > title", "Certificato Vaccinale");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T051, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "classCode", "SBADM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T052, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "moodCode", "EVN");
 		ValidationXML.checkSectionByTemplateId(errors, CVACErrorEnum.T053, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "", "2.16.840.1.113883.2.9.10.1.11.4.1", false, buildPathValue("statusCode", "code", "completed"));
 		ValidationXML.checkSectionByTemplateId(errors, CVACErrorEnum.T054, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "", "2.16.840.1.113883.2.9.10.1.11.4.2", false, buildPathValue("statusCode", "code", "cancelled"));
 		ValidationXML.equalIC(errors, CVACErrorEnum.T055, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > routeCode", "codeSystem", "2.16.840.1.113883.5.112");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T056, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > routeCode", "codeSystemName", "RouteOfAdministration");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T057, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > approachSiteCode", "codeSystem", "2.16.840.1.113883.5.1052");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T058, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > approachSiteCode", "codeSystemName", "ActSite");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T058, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > consumable", "typeCode", "CSM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T059, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystem", "2.16.840.1.113883.2.9.6.1.5");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T060, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystemName", "AIC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T061, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code > translation", "codeSystem", "2.16.840.1.113883.6.73");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T062, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > participant", "typeCode", "LOC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T063, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > entryRelationship", "typeCode", "SUBJ", "RSON", "REFR", "CAUS");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T064, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > entryRelationship > observation", "moodCode", "EVN");
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T065, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.1", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.3", buildPathValue("code", "code","30973-2"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.6.1"), buildPathValue(null, "typeCode","SUBJ"), buildPathValue(null, "inversionInd","true"), buildPathValue("code", "codeSystemName", "LOINC"));
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T066, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.1", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.4", buildPathValue("code", "code", "59778-1", "30981-5", "30980-7", "59777-3", "59778-1"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.4.642.3.308"), buildPathValue(null, "typeCode", "REFR"), buildPathValue("code", "codeSystemName", "LOINC"));
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T067, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.1", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.5", buildPathValue("code", "code","95715-9"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.6.1"), buildPathValue(null, "typeCode","RSON"), buildPathValue("code", "codeSystemName", "LOINC"));
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T068, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.1", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.6", buildPathValue("code", "code","59785-6"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.6.1"), buildPathValue(null, "typeCode","RSON"), buildPathValue("code", "codeSystemName", "LOINC"));
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T069, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.1", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.8", buildPathValue("code", "code","31044-1"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.6.1"), buildPathValue(null, "typeCode","CAUS"), buildPathValue("code", "codeSystemName", "LOINC"));
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T070, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.1", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.9", buildPathValue("code", "code","75323-6"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.6.1"), buildPathValue(null, "typeCode","RSON"), buildPathValue("code", "codeSystemName", "LOINC"));
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T071, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.2", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.10", buildPathValue("code", "code","85714-4"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.6.1"), buildPathValue(null, "typeCode","RSON"), buildPathValue("code", "displayName", "Ragione esonero/omissione o differimento"), buildPathValue("code", "codeSystemName", "LOINC"));
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T072, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.2", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.7", buildPathValue("code", "code","59784-9"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.6.1"), buildPathValue(null, "typeCode","RSON"), buildPathValue("code", "displayName", "Malattia con immunità presunta"), buildPathValue("code", "codeSystemName", "LOINC"));
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T073, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.2", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.3", buildPathValue("code", "code","30973-2"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.6.1"), buildPathValue(null, "typeCode","SUBJ"), buildPathValue(null, "inversionInd","true"), buildPathValue("code", "displayName", "Numero dose"), buildPathValue("code", "codeSystemName", "LOINC"));
 		ValidationXML.checkNestedSections(errors, CVACErrorEnum.T074, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "2.16.840.1.113883.2.9.10.1.11.4.2", "entryRelationship", "observation", "2.16.840.1.113883.2.9.10.1.11.4.9", buildPathValue("code", "code","75323-6"), buildPathValue("code", "codeSystem", "2.16.840.1.113883.6.1"), buildPathValue(null, "typeCode","RSON"), buildPathValue("code", "displayName", "Condizione"), buildPathValue("code", "codeSystemName", "LOINC"));
 		ValidationXML.equalIC(errors, CVACErrorEnum.T075, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > entryRelationship > observation > statusCode", "code", "completed");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T076, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "classCode", "SBADM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T077, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration", "moodCode", "EVN");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T078, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > consumable", "typeCode", "CSM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T079, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystem", "2.16.840.1.113883.2.9.6.1.5");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T080, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystemName", "AIC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T081, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code > translation", "codeSystem", "2.16.840.1.113883.6.73");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T082, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > participant", "typeCode", "LOC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T083, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > entryRelationship", "typeCode", "SUBJ", "RSON");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T084, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > entryRelationship > observation", "moodCode", "EVN");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T085, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > entryRelationship > observation > code", "codeSystemName", "LOINC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.T086, xml, "ClinicalDocument > component > structuredBody > component > section[ID=Certificato_Vaccinale] > entry > substanceAdministration > entryRelationship > observation > statusCode", "code", "completed");

 		return errors;
 	}
 	
 	public static List<ErrorUCTDTO> validateUC1(String xml) {
 		List<ErrorUCTDTO> errors = new ArrayList<>();
 		init(xml, errors);
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.3.2");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[id ='Certificato_Vaccinale'] > code", "code", "11369-6");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > code", "codeSystem", "2.16.840.1.113883.6.1");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > code", "codeSystemName", "LOINC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_DISPLAY_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > code", "displayName", "Storia delle immunizzazioni");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration", "classCode", "SBADM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration", "moodCode", "EVN");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_TEMPLATE_ID, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > templateID", "root", "2.16.840.1.113883.2.9.10.1.11.4.1","2.16.840.1.113883.2.9.10.1.11.4.2");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_STATUS, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > statusCode", "code", "completed" );
 		ValidationXML.hasDateFormat(errors, CVACErrorEnum.CERT_VAC_ENTRY_TIME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > effectiveTime", "value");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > routeCode", "codeSystem", "2.16.840.1.113883.5.112");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > routeCode", "codeSystemName", "HL7 RouteOfAdministration");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_APPR_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > approachSiteCode", "codeSystem", "2.16.840.1.113883.5.1052");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_APPR_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > approachSiteCode", "codeSystemName", "ActSite");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_CONS_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable", "typeCode", "CSM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_CONS_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystem", "2.16.840.1.113883.2.9.6.1.5");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_CONS_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystemName", "AIC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_PART_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > participant", "typeCode", "LOC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_PART_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > participant > participantRole", "classCode", "ROL");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "typeCode", "SUBJ","REFR","CAUS", "RSON");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_INV_ID, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "inversionInd", "true","false");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_OBS_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "classCode", "OBS");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_OBS_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "moodCode", "EVN");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_OBS_TEMPLATE_ID, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.4.3", "2.16.840.1.113883.2.9.10.1.11.4.4", "2.16.840.1.113883.2.9.10.1.11.4.5", "2.16.840.1.113883.2.9.10.1.11.4.6" ,"2.16.840.1.113883.2.9.10.1.11.4.8","2.16.840.1.113883.2.9.10.1.11.4.9");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_OBS_STATUS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > statusCode", "code", "completed");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration", "classCode", "SBADM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration", "moodCode", "EVN");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_STATUS, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > statusCode", "code", "completed");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_CONS_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable", "typeCode", "CSM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_MAN_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystem", "2.16.840.1.113883.2.9.6.1.5");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_MAN_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystemName", "AIC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_PART_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > participant", "typeCode", "LOC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_PART_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > participant > participantRole", "classCode", "ROL");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "typeCode", "SUBJ","RSON");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_INV_ID, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "inversionInd", "false", "true");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_OBS_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "classCode", "OBS");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_OBS_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "moodCode", "EVN");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_OBS_CODE_, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > code", "code","30973-2","95715-9","59785-6","31044-1","75323-6","85714-4", "59784-9"); 
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_OBS_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > code", "codeSystem", "2.16.840.1.113883.6.1");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_OBS_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > code", "codeSystemName", "LOINC");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_OBS_STATUS, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > statusCode", "code", "completed");
 		return errors;
 	}


 	public static List<ErrorUCTDTO> validateUC2(String xml) {
 		List<ErrorUCTDTO> errors = new ArrayList<>();
 		init(xml, errors);
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.3.2");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > code", "code", "11369-6");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > code", "codeSystem", "2.16.840.1.113883.6.1");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > code", "codeSystemName", "LOINC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_DISPLAY_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > code", "displayName", "Storia delle immunizzazioni");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.4.1","2.16.840.1.113883.2.9.10.1.11.4.2");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration", "classCode", "SBADM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration", "moodCode", "EVN" );
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > templateID", "root", "2.16.840.1.113883.2.9.10.1.11.4.1");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_STATUS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > statusCode", "code", "completed");
 		ValidationXML.hasDateFormat(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_TIME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > effectiveTime", "value");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > routeCode", "codeSystem", "2.16.840.1.113883.5.112");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > routeCode", "codeSystemName", "HL7 RouteOfAdministration");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_APP_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > approachSiteCode", "codeSystem", "2.16.840.1.113883.5.1052");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_APP_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > approachSiteCode", "codeSystemName", "ActSite");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_CONS_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable", "typeCode", "CSM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_CONS_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystem", "2.16.840.1.113883.2.9.6.1.5");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_CONS_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystemName", "AIC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_PART_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > participant", "typeCode", "LOC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_PART_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > participant > participantRole", "classCode", "ROL");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_ER_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "typeCode", "SUBJ", "REFR");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_ER_INV_ID, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "inversionInd", "true","false");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_ER_OBS_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "classCode", "OBS");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_ER_OBS_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "moodCode", "EVN");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_ER_OBS_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.4.3", "2.16.840.1.113883.2.9.10.1.11.4.4");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_SUB_ER_OBS_STATUS, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > statusCode", "code", "completed");
 		ValidationXML.containsIC(errors, CVACErrorEnum.SING_VAC_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "typeCode", "SUBJ", "REFR", "CAUS", "RSON");
 		ValidationXML.containsIC(errors, CVACErrorEnum.SING_VAC_INV_ID, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "inversionInd", "true", "false");
 		ValidationXML.containsIC(errors, CVACErrorEnum.SING_VAC_OBS_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "classCode", "OBS");
 		ValidationXML.containsIC(errors, CVACErrorEnum.SING_VAC_OBS_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "moodCode", "EVN");
 		ValidationXML.containsIC(errors, CVACErrorEnum.SING_VAC_OBS_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.4.3", "2.16.840.1.113883.2.9.10.1.11.4.4","2.16.840.1.113883.2.9.10.1.11.4.5","2.16.840.1.113883.2.9.10.1.11.4.6","2.16.840.1.113883.2.9.10.1.11.4.8", "2.16.840.1.113883.2.9.10.1.11.4.9");
 		ValidationXML.containsIC(errors, CVACErrorEnum.SING_VAC_OBS_STATUS, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > statusCode", "code", "completed");
 		ValidationXML.containsIC(errors, CVACErrorEnum.SING_VAC_ER_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "typeCode", "SUBJ","REFR","CAUS","RSON");
 		ValidationXML.containsIC(errors, CVACErrorEnum.SING_VAC_ER_INV_ID, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "inversionInd", "true", "false");
 		ValidationXML.containsIC(errors, CVACErrorEnum.SING_VAC_ER_OBS_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "classCode", "OBS");
 		ValidationXML.containsIC(errors, CVACErrorEnum.SING_VAC_ER_OBS_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "moodCode", "EVN");
 		ValidationXML.containsIC(errors, CVACErrorEnum.SING_VAC_ER_OBS_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.4.3", " 2.16.840.1.113883.2.9.10.1.11.4.4", "2.16.840.1.113883.2.9.10.1.11.4.5","2.16.840.1.113883.2.9.10.1.11.4.6", "2.16.840.1.113883.2.9.10.1.11.4.8", "2.16.840.1.113883.2.9.10.1.11.4.9");
 		ValidationXML.containsIC(errors, CVACErrorEnum.SING_VAC_ER_OBS_STATUS, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > statusCode", "code", "completed");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_OBS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > code", "code", "30973-2", "95715-9", "59785-6","75323-6");

 		return errors;
 	}

 	public static List<ErrorUCTDTO> validateUC3(String xml) {
 		List<ErrorUCTDTO> errors = new ArrayList<>();
 		init(xml, errors);
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.3.2");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.4.1", "2.16.840.1.113883.2.9.10.1.11.4.2");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > code", "code", "11369-6");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > code", "codeSystem", "2.16.840.1.113883.6.1");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > code", "codeSystemName", "LOINC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_DISPLAY_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > code", "displayName", "Storia delle immunizzazioni");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration", "moodCode", "EVN" );
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > templateID", "root", "2.16.840.1.113883.2.9.10.1.11.4.1", "2.16.840.1.113883.2.9.10.1.11.4.2");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_SUB_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > statusCode", "code", "completed" );
 		ValidationXML.hasDateFormat(errors, CVACErrorEnum.CERT_VAC_SUB_TIME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > effectiveTime", "value");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > routeCode", "codeSystem", "2.16.840.1.113883.5.112");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > routeCode", "codeSystemName", "HL7 RouteOfAdministration");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_APPR_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > approachSiteCode", "codeSystem", "2.16.840.1.113883.5.1052");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_APPR_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > approachSiteCode", "codeSystemName", "ActSite");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable", "typeCode", "CSM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_MAN_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystem", "2.16.840.1.113883.2.9.6.1.5");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_MAN_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystemName", "AIC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_PART_PART_ROLE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > participant > participantRole", "classCode", "ROL");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_INV_ID, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "inversionInd", "true", "false");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration", "classCode", "SBADM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_CONS_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable", "typeCode", "CSM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_CONS_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystem", "2.16.840.1.113883.2.9.6.1.5");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_CONS_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystemName", "AIC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_PART_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > participant > participantRole", "classCode", "ROL");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "typeCode", "SUBJ","REFR","CAUS", "RSON");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_OBS_STATUS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > statusCode", "code", "completed");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_PART_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > participant", "typeCode", "LOC");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_OBS_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "classCode", "OBS");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_OBS_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "moodCode", "EVN");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_OBS_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.4.3", "2.16.840.1.113883.2.9.10.1.11.4.4", "2.16.840.1.113883.2.9.10.1.11.4.5", "2.16.840.1.113883.2.9.10.1.11.4.6", "2.16.840.1.113883.2.9.10.1.11.4.8", "2.16.840.1.113883.2.9.10.1.11.4.9", "2.16.840.1.113883.2.9.10.1.11.4.10", "2.16.840.1.113883.2.9.10.1.11.4.7");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_OBS_STATUS, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > statusCode", "code", "completed");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_SUB_OBS_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "classCode", "OBS");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_SUB_OBS_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "moodCode", "EVN");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_SUB_OBS_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.4.3", "2.16.840.1.113883.2.9.10.1.11.4.4", "2.16.840.1.113883.2.9.10.1.11.4.5", "2.16.840.1.113883.2.9.10.1.11.4.6", "2.16.840.1.113883.2.9.10.1.11.4.8", "2.16.840.1.113883.2.9.10.1.11.4.9");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_SUB_OBS_STATUS, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > statusCode", "code", "completed");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_SUB_STATUS, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > statusCode", "code", "cancelled");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_OBS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > code", "code", "85714-4", "59784-9", "30973-2","95715-9","59785-6","31044-1","75323-6");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_OBS_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > code", "codeSystem", "2.16.840.1.113883.6.1");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_OBS_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > code", "codeSystemName", "LOINC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_OBS_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "typeCode", "SUBJ", "RSON");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_OBS_INV_ID, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "inversionInd", "false", "true");
 		return errors;
 	}

 	public static List<ErrorUCTDTO> validateUC4(String xml) {
 		List<ErrorUCTDTO> errors = new ArrayList<>();
 		init(xml, errors);
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.3.2");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ENTRY_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.4.1", "2.16.840.1.113883.2.9.10.1.11.4.2");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > code", "code", "11369-6");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > code", "codeSystem", "2.16.840.1.113883.6.1");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > code", "codeSystemName", "LOINC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_DISPLAY_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > code", "displayName", "Storia delle immunizzazioni");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration", "classCode", "SBADM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration", "moodCode", "EVN");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_STATUS, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > statusCode", "code", "completed" );
 		ValidationXML.hasDateFormat(errors, CVACErrorEnum.CERT_VAC_TIME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > effectiveTime", "value");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_ROUTE_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > routeCode", "codeSystem", "2.16.840.1.113883.5.112");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_ROUTE_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > routeCode", "codeSystemName", "HL7 RouteOfAdministration");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_APPR_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > approachSiteCode", "codeSystem", "2.16.840.1.113883.5.1052");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_APPR_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > approachSiteCode", "codeSystemName", "ActSite");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_CONS_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable", "typeCode", "CSM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_CONS_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystem", "2.16.840.1.113883.2.9.6.1.5");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_CONS_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystemName", "AIC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "classCode", "OBS");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "moodCode", "EVN");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_OBS_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.4.3", "2.16.840.1.113883.2.9.10.1.11.4.4", "2.16.840.1.113883.2.9.10.1.11.4.5", "2.16.840.1.113883.2.9.10.1.11.4.6", "2.16.840.1.113883.2.9.10.1.11.4.8", "2.16.840.1.113883.2.9.10.1.11.4.9");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "typeCode", "SUBJ", "REFR", "CAUS", "RSON");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_OBS_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "classCode", "OBS");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_STATUS, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > statusCode", "code", "completed");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_STATUS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > statusCode", "code", "completed", "cancelled");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_INV_ID, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "inversionInd", "true", "false");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_OBS_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "moodCode", "EVN");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration", "moodCode", "EVN");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration", "classCode", "SBADM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_OBS_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "classCode", "OBS");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_OBS_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation", "moodCode", "EVN");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_OBS_STATUS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > statusCode", "code", "completed");	
 		ValidationXML.hasDateFormat(errors, CVACErrorEnum.CERT_VAC_SUB_TIME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > effectiveTime", "value");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ROUTE_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > routeCode", "codeSystem", "2.16.840.1.113883.5.112");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ROUTE_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > routeCode", "codeSystemName", "HL7 RouteOfAdministration");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_APP_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > approachSiteCode", "codeSystem", "2.16.840.1.113883.5.1052");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_APP_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > approachSiteCode", "codeSystemName", "ActSite");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_CONS_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable", "typeCode", "CSM");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_CONS_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystem", "2.16.840.1.113883.2.9.6.1.5");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_CONS_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > consumable > manufacturedProduct > manufacturedMaterial > code", "codeSystemName", "AIC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_PART_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > participant", "typeCode", "LOC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_PART_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > participant > participantRole", "classCode", "ROL");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ER_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "typeCode", "SUBJ", "REFR", "CAUS", "RSON");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_ER_INV_ID, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship", "inversionInd", "true", "false");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_OBS_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.4.10", "2.16.840.1.113883.2.9.10.1.11.4.7");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_OBS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > code", "code", "85714-4", "59784-9","30973-2","95715-9","59785-6","31044-1","75323-6");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_OBS_CODE_SYSTEM, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > code", "codeSystem", "2.16.840.1.113883.6.1");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_OBS_CODE_SYSTEM_NAME, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > code", "codeSystemName", "LOINC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_OBS_STATUS, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > statusCode", "code", "completed");
 		//		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_OBS_VALUE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > value > originalText > reference", "value", "Malattia con presunta immunità");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_ER_OBS_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > entryRelationship > observation > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.4.3", "2.16.840.1.113883.2.9.10.1.11.4.4", "2.16.840.1.113883.2.9.10.1.11.4.5", "2.16.840.1.113883.2.9.10.1.11.4.6", "2.16.840.1.113883.2.9.10.1.11.4.8", "2.16.840.1.113883.2.9.10.1.11.4.9");
 		ValidationXML.containsIC(errors, CVACErrorEnum.CERT_VAC_SUB_ROOT, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > templateId", "root", "2.16.840.1.113883.2.9.10.1.11.4.2");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_PART_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > participant", "typeCode", "LOC");
 		ValidationXML.equalIC(errors, CVACErrorEnum.CERT_VAC_SUB_PART_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody > component > section[ID='Certificato_Vaccinale'] > entry > substanceAdministration > participant > participantRole", "classCode", "ROL");
 		return errors;
 	}

 	private static void init(String xml, List<ErrorUCTDTO> errors) {
 		ValidationXML.equalIC(errors, CVACErrorEnum.COMP_TYPE_CODE, xml, "ClinicalDocument > component", "typeCode", "COMP");
 		ValidationXML.equalIC(errors, CVACErrorEnum.COMP_COND_IND, xml, "ClinicalDocument > component", "contextConductionInd", "true");
 		ValidationXML.equalIC(errors, CVACErrorEnum.COMP_CLASS_CODE, xml, "ClinicalDocument > component > structuredBody", "classCode", "DOCBODY");
 		ValidationXML.equalIC(errors, CVACErrorEnum.COMP_MOOD_CODE, xml, "ClinicalDocument > component > structuredBody", "moodCode", "EVN");
 		ValidationXML.equalIC(errors, CVACErrorEnum.COMP_STRUCT_TYPE_CODE, xml, "ClinicalDocument > component > structuredBody > component", "typeCode", "COMP");
 		ValidationXML.equalIC(errors, CVACErrorEnum.COMP_COMP_COND_IND, xml, "ClinicalDocument > component > structuredBody > component", "contextConductionInd", "true");
 	}

 }
